// RiffRemover.cpp : Defines the entry point for the console application.
//

#include "stdafx.h"

#define BUFFERSIZE 4096
#define ERR_CANT_OPEN_INPUT_FILE 1
#define ERR_CANT_OPEN_OUTPUT_FILE 2
#define ERR_CANT_READ_INPUT_FILE 3
#define ERR_CANT_WRITE_OUTPUT_FILE 4
#define ERR_INPUT_NOT_A_RIFF_FILE 5
#define ERR_READ_ERROR 6
#define ERR_WRITE_ERROR 7

const char RiffID[] = "RIFF";
const char DataID[] = "data";

typedef char ID[4];
typedef struct {
  ID             chunkID;
  long           chunkSize;
} ChunkHeader;

int logLevel = 0;

int log(int _logLevel, _In_z_ _Printf_format_string_ const char * _Format, ...){
	int result = 0;

	va_list args;
	va_start(args, _Format);
	if(_logLevel >= logLevel){
		result = vprintf(_Format, args);
	}
	va_end(args);
	return result;
}

int stripRiff(_TCHAR* inputFilePath, _TCHAR* outputFilePath){
	FILE *fpIn, *fpOut;
	errno_t err;
	ChunkHeader chunkHeader;
	char buffer[BUFFERSIZE];
	size_t copiedSize;
	long copied;
	int remainder;
	int result = 0;

	if((err = _wfopen_s(&fpIn, inputFilePath, L"rb")) != 0){
		result = ERR_CANT_OPEN_INPUT_FILE;
		goto end;
	}

	if((err = _wfopen_s(&fpOut, outputFilePath, L"wb")) != 0){
		result = ERR_CANT_OPEN_OUTPUT_FILE;
		goto closeIn;
	}

	if(fread(&chunkHeader, sizeof(ChunkHeader), 1, fpIn) != 1){
		result = ERR_CANT_READ_INPUT_FILE;
		goto closeOut;
	}

	if(memcmp(&chunkHeader.chunkID, RiffID, sizeof(ID)) != 0){
		return ERR_INPUT_NOT_A_RIFF_FILE;
	}
	fseek(fpIn, 4, SEEK_CUR);

	while(fread(&chunkHeader, sizeof(ChunkHeader), 1, fpIn) == 1){
		if(memcmp(&chunkHeader.chunkID, DataID, sizeof(ID)) == 0){
			log(0, "Found data chunck, size: %i bytes\n", chunkHeader.chunkSize);
			copied = 0;
			while((chunkHeader.chunkSize - copied) >= BUFFERSIZE){
				copiedSize = fread(&buffer, 1, BUFFERSIZE, fpIn);
				if(copiedSize != BUFFERSIZE){
					result = ERR_READ_ERROR;
					goto closeOut;
				}
				if(fwrite(&buffer, 1, copiedSize, fpOut) != copiedSize){
					result = ERR_WRITE_ERROR;
					goto closeOut;
				}
				copied += copiedSize;
			}
			remainder = (chunkHeader.chunkSize - copied);
			if((chunkHeader.chunkSize - copied) > 0){
				if(fread(&buffer, 1, remainder, fpIn) != remainder){
					result = ERR_READ_ERROR;
					goto closeOut;
				}
				if(fwrite(&buffer, 1, remainder, fpOut) != remainder){
					result = ERR_WRITE_ERROR;
					goto closeOut;
				}
			}
		} else {
			log(0, "Skipping chunk: %c%c%c%c\n", chunkHeader.chunkID[0], chunkHeader.chunkID[1], chunkHeader.chunkID[2], chunkHeader.chunkID[3]);
			fseek(fpIn, chunkHeader.chunkSize, SEEK_CUR);
		}
	}

	closeOut:
		fclose(fpOut);
	closeIn:
		fclose(fpIn);
	end:
		return result;
}

int _tmain(int argc, _TCHAR* argv[])
{
	int result;

	if(argc < 3){
		printf("Please specify a file to strip and an output file\n");
		exit(1);
	}

	if(result = stripRiff(argv[1], argv[2])){
		switch(result){
			case ERR_CANT_OPEN_INPUT_FILE:
				printf("Can't open input file %s\n",argv[1]);
				break;
			case ERR_CANT_OPEN_OUTPUT_FILE:
				printf("Can't open output file %s, error: %i\n", argv[2]);
				break;
			case ERR_CANT_READ_INPUT_FILE:
				printf("Can't read input file %s\n", argv[1]);
				break;
			case ERR_INPUT_NOT_A_RIFF_FILE:
				printf("Input file %s is not a RIFF file\n", argv[1]);
				break;
			case ERR_READ_ERROR:
				printf("Can't read input file %s\n",argv[1]);
				break;
			case ERR_WRITE_ERROR:
				printf("Can't write output file %s", argv[2]);
				break;
		}
		return result;
	}
	return 0;
}

